/********************************************************************
 * (C) Copyright 1998 by Hewlett-Packard GmbH. All rights reserved. *
 ********************************************************************/

/* ***********************************************************************
   *                                                                     *
   * decode.c                                                            *
   * Christian Zurhorst, BID R&D, 13. Nov. 1997                          *
   *                                                                     *
   * functions to control configuration space and target decoders        *
   *                                                                     *
   ***********************************************************************

*/

#if defined(_MSC_VER)
# pragma data_seg("decode_c","FAR_DATA")
#elif defined(__BORLANDC__)
# pragma option -dc
#endif /* _MSC_VER etc. */

#ifdef BEST_FIRMWARE

# include <capitype.h>

#else

# include <typedefs.h>
/* TODO: remove this as soon as bdprpget has found a good place */
#include <mini_api.h>

#endif  /* BEST_FIRMWARE */


#include <board.h>
#include <decode.h>
#include <dynamic.h>
#include <errcapi.h>
#include <iocommon.h>
#include <regconst.h>
#include <regx20.h>
#include <regx21.h>
#include <regx24.h>
#include <regx25.h>
#include <regx50.h>
#include <regx51.h>
#include <regx52.h>
#include <regx53.h>
#include <session.h>
#include <target.h>


#define BAR0	0x10
/*
#define BAR1	0x14
#define BAR2	0x18
#define BAR3	0x1c
#define BAR4	0x20
*/
#define BAR5	0x24
#define BARX	0x30


#define STD_BASEADDR(p)		((p >= BAR0) && (p <= BAR5))
#define XROM_BASEADDR(p)	(p == BARX)
#define STD_DECODER(p)		((b_decodertype)((p - BAR0) / 0x4 + 1))
/*
#define STD_BAR(p)		((p - 1) * 0x4 + BAR0)
*/

/*
#define DEC_RESULT_BASE_SIZE_MISMATCH  0x01
#define DEC_RESULT_INVALID_SIZE        0x02
#define DEC_RESULT_INVALID_MODE        0x03
#define DEC_RESULT_CONFIG_ACCESS_ERROR 0x04
#define DEC_RESULT_HW_BASE_WRITE_ERROR 0x05
#define DEC_RESULT_INVALID_DECODER     0x06
#define DEC_RESULT_BASE_NOT_0          0x07
#define DEC_RESULT_ROM_SIZE_0_ENABLED  0x08
#define DEC_RESULT_HW_BUSY             0x09
*/

#ifndef E2921A_MIN

/* avoid compiler warnings */
static const int Abort = 0;

static const b_errtype decoder_result[] = {
  B_E_DEC_CHECK,
  B_E_DEC_SIZE_BASE_MISMATCH,
  B_E_DEC_INVALID_SIZE,
  B_E_DEC_INVALID_MODE,
  B_E_DEC_CONFIG_ACCESS,
  B_E_DEC_BASE_WRITE,
  B_E_DEC_INVALID_DECODER,
  B_E_DEC_BASE_NOT_0,
  B_E_DEC_ROM_SIZE_0_ENABLED,
  B_E_HW_BUSY
};


static unsigned long Log2(unsigned long inval)
{
  unsigned long outval;
  for (outval = 0; (0 != (inval >>= 1)); )
    outval++;

  return outval;
}

static b_errtype valid_decoder_num(b_handletype handle, b_int32 decoder)
{
  return (BestParamCheck(handle, B_PARAM_DECODER_GEN, decoder, 0UL));
}


static b_int32 mask2size(b_int32 mask)
{
  b_int32 size;
  size = (b_int32) Log2(~mask + 1);

  /* check if mask is valid (no holes) */
  if ((b_int32) (size ? 1UL << (int) size : 0UL) != (~mask + 1UL))
    size = B_DEC_INVALID;

  return size;
}


static b_errtype pushpreg(b_handletype handle)
{
  return BestBasicCommand(handle, CMD_EXE_TDEC_PREPREG_PUSH,
    NULL, 0,
    NULL, NULL);
}


static b_errtype poppreg(b_handletype handle)
{
  return BestBasicCommand(handle, CMD_EXE_TDEC_PREPREG_POP,
    NULL, 0,
    NULL, NULL);
}


static b_errtype low_base_set(b_handletype handle,
    b_int32 value, b_int32 mask,
    b_int32 addsize)
{
  B_DECLARE_FUNCNAME ("low_base_set (internal)");
  B_TRY_VARS_NO_PROG;
  b_int32 prop;
  B_TRY_BEGIN
  {
    /* 'prop' holds the value of the behavior property */
    B_TRY(BestTargetDecoderPropGet(handle, B_DEC_BEHAVIOR, &prop));

    switch (prop)
    {
    case B_BEH_NORMAL:
      /* 'prop' holds now the value of the size property */
      if (mask)
      {
        prop = mask2size(mask);
      }

      else
        prop = 32;

      prop += addsize;
      B_TRY(BestTargetDecoderPropSet(handle, B_DEC_BASEADDR, value & mask));
      B_TRY(BestTargetDecoderPropSet(handle, B_DEC_SIZE, prop));
      B_TRY(BestTargetDecoderPropSet(handle, B_DEC_LOCATION, value & 0x07UL));
      B_TRY(BestTargetDecoderPropSet(handle, B_DEC_PREFETCH,
          value & 0x08UL ? 1UL : 0UL));
      break;

    case B_BEH_CONFIG:
      B_TRY(BestTargetDecoderPropSet(handle, B_DEC_BASEADDR, value));
      B_TRY(BestTargetDecoderPropSet(handle, B_DEC_MASK, mask));
      break;

    case B_BEH_CUSTOM:
      /* 'prop' holds the location value */
      prop = value & 0x01UL ? B_LOC_IO : (value & 0x06UL) >> 1;
      B_TRY(BestTargetDecoderPropSet(handle, B_DEC_BASEADDR,
          value &
          (1L << (prop == B_LOC_IO ? 2 : 4)) - 1));
      B_TRY(BestTargetDecoderPropSet(handle, B_DEC_MASK, mask));
      B_TRY(BestTargetDecoderPropSet(handle, B_DEC_LOCATION,
          value & 0x07UL));
      B_TRY(BestTargetDecoderPropSet(handle, B_DEC_PREFETCH,
          value & 0x08UL ? 1UL : 0UL));
      break;

    default:
      BestLastErrorParamSet(handle, B_ERRPAR_1, (b_int32)B_PARAM_DECODER_STD);
      BestLastErrorParamSet(handle, B_ERRPAR_2, (b_int32)B_DEC_BEHAVIOR);
      BestLastErrorParamSet(handle, B_ERRPAR_3, prop);
      B_TRY_ERROR(B_E_VALUE);
      break;
    }
  }
  B_ERRETURN(B_TRY_RET);
}


#ifndef BEST_FIRMWARE
/* should be in firmware but requires more modules, not needed yet */

static b_errtype std_base_set(b_handletype handle,
    b_decodertype decoder,
    b_int32 value, b_int32 mask)
{
  B_TRY_VARS;
  b_int32 lo_dec, size, prop;
  /* first thing is to find out whether the base address refers to a 32-bit
   * decoder or whether it is the lower or high part of a 64-bit decoder. */

  switch (decoder)
  {
  case B_DEC_STANDARD_1:
  case B_DEC_STANDARD_3:
  case B_DEC_STANDARD_5:
    lo_dec = (b_int32)decoder;
    break;

  case B_DEC_STANDARD_2:
  case B_DEC_STANDARD_4:
  case B_DEC_STANDARD_6:
    lo_dec = (b_int32)(decoder - 1);
    break;

  default:
    assert(Abort && "Function std_base_set can be used for standard decoders only!");
    break;
  } /*lint !e788 ... not all enums used */

  B_TRY_BEGIN
  {
    /* save the contents of the preparation registers */
    B_TRY_PROGRESS(pushpreg(handle));

    /* read the 64-bit base decoder in the preparation registers */
    B_TRY(BestTargetDecoderRead(handle, (b_decodertype) lo_dec));

    /* 'prop' holds the value of the DAC property of the lower part of the
     * addressed imaginary 64-bit decoder */
    B_TRY(BestTargetDecoderPropGet(handle, B_DEC_DAC, &prop));

    switch (prop)
    {
    case B_DAC_NO:
      /* it's a 32-bit decoder */
      if ((b_int32) decoder != lo_dec)
      {
        /* we have read the wrong decoder, now get the right one */
        B_TRY(BestTargetDecoderRead(handle, decoder));
      }

      /* low_base_set will add 32, if decoder is off do -32 */
      size = (mask ? 0 : 0xffffffe0UL);
      (void)low_base_set(handle, value, mask, size);
      break;

    case B_DAC_YES:
      if ((b_int32) decoder == lo_dec)
      {
        /* 'prop' holds the value of the behavior property */
        B_TRY(BestTargetDecoderPropGet(handle, B_DEC_BEHAVIOR, &prop));

        switch (prop)
        {
        case B_BEH_NORMAL:
          /* 'prop' holds the size property */
          B_TRY(BestTargetDecoderPropGet(handle, B_DEC_SIZE, &prop));
          size = (prop >= 32 ? prop - 32 : 0);
          break;

        default:
          /* size is the additional size of a potentially 64-bit decoder */
          B_TRY(BestTargetDecoderPropGet(handle, B_DEC_SIZE, &prop));
          size = 0;
          break;
        }

        /* change the lower base register of a 64-bit decoder */
        /* low_base_set will add 32, if decoder goes off do -32 */
        size = (size || mask) ? size : 0xffffffe0UL;
        (void)low_base_set(handle, value, mask, size);
      }

      else
      {
        /* change the higher base register of a 64-bit decoder */
        decoder = (b_decodertype) lo_dec;

        /* 'prop' holds the value of the behavior property */
        B_TRY(BestTargetDecoderPropGet(handle, B_DEC_BEHAVIOR, &prop));

        switch (prop)
        {
        case B_BEH_NORMAL:
          /* 'prop' holds the value of the size property */
          B_TRY(BestTargetDecoderPropGet(handle, B_DEC_SIZE, &prop));

          if (prop < 32)
          {
            /* size is < 32 => mask (himask here!) has to be 0xffffffff to be
             * compliant */
            BestLastErrorParamSet(handle, B_ERRPAR_1, mask);
            BestLastErrorParamSet(handle, B_ERRPAR_2,
              4UL * (b_int32)(decoder - B_DEC_STANDARD_1) + 0x10UL);
            B_TRY_FAIL(mask == 0xffffffffUL ? B_E_OK : B_E_CONFIG_MASK_INVALID);
          }

          else
          {
            size = mask2size(mask);
            /* set mask in all lower half bits */
            prop = (size == B_DEC_INVALID ? size : 32 + size);
          }

          B_TRY(BestTargetDecoderPropSet(handle, B_DEC_SIZE, prop));
          B_TRY(BestTargetDecoderPropSet(handle, B_DEC_BASEADDR_HI,
              value & mask));
          break;

        case B_BEH_CONFIG:
        case B_BEH_CUSTOM:
          B_TRY(BestTargetDecoderPropSet(handle, B_DEC_BASEADDR_HI, value));
          B_TRY(BestTargetDecoderPropSet(handle, B_DEC_MASK_HI, mask));
          break;

        default:
          BestLastErrorParamSet(handle, B_ERRPAR_1, (b_int32)B_PARAM_DECODER_STD);
          BestLastErrorParamSet(handle, B_ERRPAR_2, (b_int32)B_DEC_BEHAVIOR);
          BestLastErrorParamSet(handle, B_ERRPAR_3, prop);
          B_TRY_ERROR(B_E_VALUE);
          break;
        }
      }

      break;

    case B_DAC_BOTH:
      BestLastErrorParamSet(handle, B_ERRPAR_1, (b_int32)B_PARAM_DECODER_STD);
      BestLastErrorParamSet(handle, B_ERRPAR_2, (b_int32)B_DEC_DAC);
      BestLastErrorParamSet(handle, B_ERRPAR_3, (b_int32)B_DAC_NO);
      BestLastErrorParamSet(handle, B_ERRPAR_4, (b_int32)B_DAC_YES);
      BestLastErrorParamSet(handle, B_ERRPAR_5, (b_int32)B_DAC_BOTH);
      B_TRY_ERROR(B_E_RANGE);
      break;

    default:
      assert(Abort && "Unanticipated decoder prop in std_base_set()");
      break;
    }

    B_TRY_PROGRESS(BestTargetDecoderProg(handle, decoder));
    B_TRY_ERROR(B_E_INTERNAL_RETURN);
  }

  B_TRY_CATCH
  {
    B_TRY_PASSED
    {
      (void)poppreg(handle);
    }

    B_TRY_PASSED
    {
      B_TRY_RET = B_E_OK;
    }
  }

  return B_TRY_RET;
}

#endif  /* BEST_FIRMWARE */


#ifndef BEST_FIRMWARE
/* should be in firmware but requires more modules, not needed yet */

static b_errtype xrom_base_set(b_handletype handle,
    b_int32 value, b_int32 mask)
{
  B_TRY_VARS;
  b_int32 prop;
  B_TRY_BEGIN
  {
    /* save the contents of the preparation registers */
    B_TRY_PROGRESS(pushpreg(handle));

    /* 'prop' holds the value of the behavior property */
    B_TRY(BestTargetDecoderPropGet(handle, B_DEC_BEHAVIOR, &prop));

    switch (prop)
    {
    case B_BEH_NORMAL:
      B_TRY(BestTargetDecoderPropSet(handle, B_DEC_BASEADDR,
          value & mask));
      B_TRY(BestTargetDecoderPropSet(handle, B_DEC_SIZE,
				     mask2size(mask & ~0x1UL)));
      break;

    case B_BEH_CUSTOM:
      B_TRY(BestTargetDecoderPropSet(handle, B_DEC_BASEADDR, value));
      B_TRY(BestTargetDecoderPropSet(handle, B_DEC_MASK, mask));
      break;

    default:
      BestLastErrorParamSet(handle, B_ERRPAR_1,
			    (b_int32)B_PARAM_DECODER_EXP);
      BestLastErrorParamSet(handle, B_ERRPAR_2, (b_int32)B_DEC_BEHAVIOR);
      BestLastErrorParamSet(handle, B_ERRPAR_3, prop);
      B_TRY_ERROR(B_E_VALUE);
      break;
    }

    B_TRY_PROGRESS(BestTargetDecoderProg(handle, B_DEC_EXPROM));
    B_TRY_ERROR(B_E_INTERNAL_RETURN);
  }

  B_TRY_CATCH
  {
    B_TRY_PASSED
    {
      (void)poppreg(handle);
    }

    B_TRY_PASSED
    {
      B_TRY_RET = B_E_OK;
    }
  }

  return B_TRY_RET;
}

#endif  /* BEST_FIRMWARE */


/* -----------------------------------------------------------------
 * Configuration Space Programming Functions
 * ----------------------------------------------------------------- */

#ifndef BEST_FIRMWARE
/* should be in firmware but requires more modules, not needed yet */

b_errtype EXPORT BestConfRegSet(
    b_handletype handle,
    b_int32 offset,
    b_int32 value
)
{
  B_DECLARE_FUNCNAME("BestConfRegSet [conrset]");

  B_TRY_VARS_NO_PROG;

  B_TRY_BEGIN
  {
    /* license checking */
    B_TRY_LICENSE(B_CAPABILITY_EXERCISER);

    B_TRY_FCT_PARAM(2, offset > 0x3c);
    B_TRY_FAIL(offset & 0x03UL ? B_E_NO_DWORD_BOUNDARY : B_E_OK);

    if (Best16BitRegisterFile(handle))
    {
      if (STD_BASEADDR(offset))
      {
        b_int32 mask;
        /* it's a standard range decoder base address, so reduce the problem
         * to find the according decoder properties */
        B_TRY(BestConfRegMaskGet(handle, offset, &mask));
        B_TRY(std_base_set(handle, STD_DECODER(offset), value, mask));
      }

      else if (XROM_BASEADDR(offset))
      {
        b_int32 mask;
        /* it's an expansion ROM decoder base address */
        B_TRY(BestConfRegMaskGet(handle, offset, &mask));
        B_TRY(xrom_base_set(handle, value, mask));
      }

      else
      {
        B_TRY(BestAbstractPropLongSet(handle, CMD_EXE_CONFREG_SET, (b_int8) offset, value));
      }
    }

    else
    {
      b_errtype err;
      b_int8 zw8;
      B_TRY(BestBasicBlockWrite(handle, CONF_SPACE_OFFS, (b_int8ptr)&offset, 4UL));
      err = BestBasicBlockWrite(handle, CONF_SPACE_DATA, (b_int8ptr)&value, 4UL);

      if (err == B_E_FUNC)
      {
        /* an error occured, so get the result code */
        B_TRY(BestBasicBlockRead(handle, T_DEC_RESULT, &zw8, 1UL));

        if (zw8 > 0x09)
        {
          B_TRY_ERROR(B_E_DEC_CHECK);
        }

        else
        {
          B_TRY_ERROR(decoder_result[zw8]);
        }
      }
    }
  }

  B_ERRETURN(B_TRY_RET);
}

#endif  /* BEST_FIRMWARE */


#ifndef BEST_FIRMWARE
/* should be in firmware but requires more modules, not needed yet */

b_errtype EXPORT BestConfRegModify(
    b_handletype handle,
    b_int32 offset,
    b_int32 value,
    b_int32 mask
)
{
  B_DECLARE_FUNCNAME("BestConfRegModify [conrmodify]");

  B_TRY_VARS_NO_PROG;

  B_TRY_BEGIN
  {
    /* license checking */
    B_TRY_LICENSE(B_CAPABILITY_EXERCISER);

    B_TRY_FCT_PARAM(2, offset > 0x3c);
    B_TRY_FAIL(offset & 0x03UL ? B_E_NO_DWORD_BOUNDARY : B_E_OK);

    if (Best16BitRegisterFile(handle))
    {
      if (STD_BASEADDR(offset) || XROM_BASEADDR(offset))
      {
        b_int32 baseaddr;
        /* it's a standard range decoder base address, so reduce the problem
         * to find the according decoder properties */
        B_TRY(BestConfRegGet(handle, offset, &baseaddr));
        B_TRY(BestConfRegSet(handle, offset, (mask & value) | (~mask & baseaddr)));
      }

      else
      {
        b_int8 buf[IN_EXE_CONFREG_MODIFY];
        b_int8ptr bufptr;
        b_int8 zw8;
        zw8 = (b_int8) offset;
        bufptr = BestByteCopy((b_int8ptr) buf, &zw8, 1UL);
        bufptr = BestLong2Stream(bufptr, &value, 1UL);
        bufptr = BestLong2Stream(bufptr, &mask, 1UL);

        B_TRY(BestBasicCommand(handle, CMD_EXE_CONFREG_MODIFY,
            buf, IN_EXE_CONFREG_MODIFY,
            NULL, NULL));
      }
    }

    else
    {
      b_int32 baseaddr;
      /* for the E2925A let's do it the conventional way */
      B_TRY(BestConfRegGet(handle, offset, &baseaddr));
      B_TRY(BestConfRegSet(handle, offset, (mask & value) | (~mask & baseaddr)));
    }
  }

  B_ERRETURN(B_TRY_RET);
}

#endif  /* BEST_FIRMWARE */

#endif  /* E2921A_MIN */


b_errtype EXPORT BestConfRegGet(
    b_handletype handle,
    b_int32 offset,
    b_int32 * value
)
{
  B_DECLARE_FUNCNAME("BestConfRegGet [conrget]");

  B_TRY_VARS_NO_PROG;

  B_TRY_BEGIN
  {
    /* license checking */
    B_TRY_LICENSE(B_CAPABILITY_EXERCISER);

    B_TRY_FCT_PARAM(2, offset > 0x3c);
    B_TRY_FAIL(offset & 0x03UL ? B_E_NO_DWORD_BOUNDARY : B_E_OK);

    if (Best16BitRegisterFile(handle))
    {
      B_TRY(BestAbstractPropLongGet(handle, CMD_EXE_CONFREG_GET, (b_int8) offset, value));
    }

    else
    {
      B_TRY(BestBasicBlockWrite(handle, CONF_SPACE_OFFS,
          (b_int8 *) & offset, 4UL));
      B_TRY(BestBasicBlockRead(handle, CONF_SPACE_DATA, (b_int8 *) value, 4UL));
    }
  }

  B_ERRETURN(B_TRY_RET);
}


#ifndef E2921A_MIN

#ifndef BEST_FIRMWARE
/* should be in firmware but requires more modules, not needed yet */

b_errtype EXPORT BestConfRegMaskSet(
    b_handletype handle,
    b_int32 offset,
    b_int32 value
)
{
  B_DECLARE_FUNCNAME("BestConfRegMaskSet [conrmaskset]");

  B_TRY_VARS_NO_PROG;

  B_TRY_BEGIN
  {
    /* license checking */
    B_TRY_LICENSE(B_CAPABILITY_EXERCISER);

    B_TRY_FCT_PARAM(2, offset > 0x3c);
    B_TRY_FAIL(offset & 0x03UL ? B_E_NO_DWORD_BOUNDARY : B_E_OK);

    if (Best16BitRegisterFile(handle))
    {
      if (STD_BASEADDR(offset))
      {
        b_int32 reg;
        /* it's a standard range decoder base address, so reduce the problem
         * to find the according decoder properties */
        B_TRY(BestConfRegGet(handle, offset, &reg));
        B_TRY(std_base_set(handle, STD_DECODER(offset), reg, value));
      }

      else if (XROM_BASEADDR(offset))
      {
        b_int32 reg;
        /* it's an expansion ROM decoder base address */
        B_TRY(BestConfRegGet(handle, offset, &reg));
        B_TRY(xrom_base_set(handle, reg, value));
      }

      else
      {
        B_TRY(BestAbstractPropLongSet(handle, CMD_EXE_CONFMASK_SET, (b_int8) offset, value));
      }
    }

    else
    {
      b_errtype err;
      b_int8 zw8;
      b_int32 zw32;
      if (offset == 0x30UL)
      {
        /* rom decoder */
        if ((value & 0xFFFFFFFEUL) != 0)
        {
          B_TRY(BestBoardPropGet(handle, B_BOARD_ROMUSAGE, &zw32));
          B_TRY_FAIL(zw32 == B_ROMUSAGE_INTERNAL ? B_E_ROM_INTERNAL : B_E_OK);
        }
      }

      B_TRY(BestBasicBlockWrite(handle, CONF_SPACE_OFFS,
          (b_int8ptr)&offset, 4UL));

      err = BestBasicBlockWrite(handle, CONF_SPACE_MASK,
        (b_int8ptr)&value, 4UL);

      if (err == B_E_FUNC)
      {
        /* an error occured, so get the result code */
        B_TRY(BestBasicBlockRead(handle, T_DEC_RESULT, &zw8, 1UL));

        if (zw8 > 0x09)
        {
          B_TRY_ERROR(B_E_DEC_CHECK);
        }

        else
        {
          B_TRY_ERROR(decoder_result[zw8]);
        }
      }
    }
  }

  B_ERRETURN(B_TRY_RET);
}

#endif  /* BEST_FIRMWARE */


b_errtype EXPORT BestConfRegMaskGet(
    b_handletype handle,
    b_int32 offset,
    b_int32 * value
)
{
  B_DECLARE_FUNCNAME("BestConfRegMaskGet [conrmaskget]");

  B_TRY_VARS_NO_PROG;

  B_TRY_BEGIN
  {
    /* license checking */
    B_TRY_LICENSE(B_CAPABILITY_EXERCISER);

    B_TRY_FCT_PARAM(2, offset > 0x3c);
    B_TRY_FAIL(offset & 0x03UL ? B_E_NO_DWORD_BOUNDARY : B_E_OK);

    if (Best16BitRegisterFile(handle))
    {
      B_TRY(BestAbstractPropLongGet(handle, CMD_EXE_CONFMASK_GET, (b_int8) offset, value));
    }

    else
    {
      B_TRY(BestBasicBlockWrite(handle, CONF_SPACE_OFFS, (b_int8ptr) & offset, 4UL));
      B_TRY(BestBasicBlockRead(handle, CONF_SPACE_MASK, (b_int8ptr) value, 4UL));
    }
  }

  B_ERRETURN(B_TRY_RET);
}






/* ------------------------------------------------------------------------
 * PCI Target Programming Functions
 * ------------------------------------------------------------------------ */

#ifndef BEST_FIRMWARE

b_errtype EXPORT BestTargetDecoderPropDefaultSet(b_handletype handle,
    b_decodertype decoder_num)
{
  B_TRY_VARS_NO_PROG;
  b_param_infotype *ParamInfo;
  b_int32 i;
  /* Generic info for range checking */
  b_generic_infotype *GenInfo;
  b_paramtype DecParam;
  B_TRY_BEGIN
  {
    /* license checking */
    B_TRY_LICENSE(B_CAPABILITY_HOSTINT);

    /* Need to convert b_decodertype into b_paramtype */
    B_TRY(BestParamInfoGet(handle, B_PARAM_DECODER_GEN,
        (b_int32)decoder_num, &ParamInfo, (b_int32)B_ENUM_SEARCH));

    assert (ParamInfo);
    
#if 0
    if (ParamInfo)
      DecParam = ParamInfo->grouploop.param;
    else
      B_TRY_FAIL(B_E_FUNC);
#endif

    DecParam = ParamInfo->grouploop.param;

    /* Get pointer to generic info */
    B_TRY(BestGenInfoGet(handle, DecParam, &GenInfo));

    for (i = 0; i < GenInfo->num_elem; i++)
    {
      /* Get pointer to i-th (existing) property */
      B_TRY(BestParamInfoGet(handle, DecParam, i, &ParamInfo,
          (b_int32)B_INDEX_SEARCH));

      switch (ParamInfo->proptyp.decprop)
      {
        /* the following properties are obsolete, so we do not touch them */
      case B_DEC_MODE:
        break;

      default:
        /* Set it to default */
        B_TRY(BestTargetDecoderPropSet(handle, ParamInfo->proptyp.decprop,
            ParamInfo->defaultval));
        break;
      } /*lint !e788 ... not all enums used */
    }
  }

  B_ERRETURN(B_TRY_RET);
}

#endif  /* BEST_FIRMWARE */


/* --------------------------------------------------------------------------
 *
 * -------------------------------------------------------------------------- */

b_errtype EXPORT BestTargetDecoderPropSet(
    b_handletype handle,
    b_decproptype decprop,
    b_int32 value
)
{
  B_DECLARE_FUNCNAME("BestTargetDecoderPropSet [tdprpset]");

  b_int8 reg8_val;
  b_errtype err;
  /* license checking */
  B_LICENSECHECK(B_CAPABILITY_HOSTINT);

  /* because we do not know the decoder we cannot do the range checking */

  if (Best16BitRegisterFile(handle))
  {
	if (BestHasIchiban20(handle) &&
	    decprop==B_DEC_SPEED &&
		(value==B_DSP_FAST || value==B_DSP_PROTECTEDFAST)
	   )
	{
      /* Ichiban 2.0 restriction for all decoders */
	  BestLastErrorParamSet(handle, B_ERRPAR_3, value);
	  err=B_E_VALUE;
	}
	else
	{  
      err=BestAbstractPropLongSet(handle, CMD_EXEFHIF_TDECPROP_SET, (b_int8) decprop, value);
    }
  }
  else
  {
    switch (decprop)
    {
    case B_DEC_MODE:
      reg8_val = (b_int8) value;

      switch (reg8_val)
      {
      case B_MODE_MEM:
      case B_MODE_IO:
      case B_MODE_CONFIG:
        err = BestBasicBlockWrite(handle, T_DEC_MODE, &reg8_val, 1UL);
        break;

      default:
	B_FCT_PARAM_ERROR(value, "Unknown mode");
      }

      break;

    case B_DEC_SIZE:
      reg8_val = (b_int8) value;
      err = BestBasicBlockWrite(handle, T_DEC_SIZE, &reg8_val, 1UL);
      break;

    case B_DEC_BASEADDR:
      err = BestBasicBlockWrite(handle, T_DEC_BASE, (b_int8 *) & value, 4UL);
      break;

    case B_DEC_SPEED:
      reg8_val = (b_int8) value;

      B_FCT_PARAM_CHK(3, ((reg8_val != B_DSP_MEDIUM) && (reg8_val != B_DSP_SLOW)));

      err = BestBasicBlockWrite(handle, T_DEC_SPEED, &reg8_val, 1UL);

      break;

    case B_DEC_LOCATION:
      reg8_val = (b_int8) value;

      switch (reg8_val)
      {
      case B_LOC_SPACE32:
      case B_LOC_SPACE64:
      case B_LOC_BELOW1MEG:
        err = BestBasicBlockWrite(handle, T_DEC_LOCATION, &reg8_val, 1UL);
        break;

      case B_LOC_IO:
        reg8_val = B_MODE_IO;
        err = BestBasicBlockWrite(handle, T_DEC_MODE, &reg8_val, 1UL);
        break;

      default:
	B_FCT_PARAM_ERROR(value, "Unknown Location");
      }

      break;

    case B_DEC_PREFETCH:
      reg8_val = (b_int8) value;

      B_FCT_PARAM_CHK_R(value, ((reg8_val != 0) && (reg8_val != 1)), "");

      err = BestBasicBlockWrite(handle, T_DEC_PREFETCH, &reg8_val, 1UL);

      break;

    default:
      B_FCT_PARAM_ERROR(decprop, "Invalid property");
    }                           /*lint !e788 ... not all enums.. switch decprop */
  }                             /* 8-bit register file */

  B_ERRETURN(err);
}


b_errtype EXPORT BestTargetDecoderPropGet(
    b_handletype handle,
    b_decproptype decprop,
    b_int32 * value
)
{
  B_DECLARE_FUNCNAME("BestTargetDecoderPropGet [tdprpget]");

  b_errtype err;
  b_int32 reg_val;
  b_int8 reg8_val;
  /* license checking */
  B_LICENSECHECK(B_CAPABILITY_HOSTINT);

  if (Best16BitRegisterFile(handle))
  {
    err = BestAbstractPropLongGet(handle, CMD_EXEFHIF_TDECPROP_GET, (b_int8) decprop, value);
  }
  else
  {
    switch (decprop)
    {
    case B_DEC_MODE:
      err = BestBasicBlockRead(handle, T_DEC_MODE, &reg8_val, 1UL);
      *value = (b_int32) reg8_val;
      break;

    case B_DEC_SIZE:
      err = BestBasicBlockRead(handle, T_DEC_SIZE, &reg8_val, 1UL);
      *value = (b_int32) reg8_val;
      break;

    case B_DEC_BASEADDR:
      err = BestBasicBlockRead(handle, T_DEC_BASE, (b_int8 *) & reg_val, 4UL);
      *value = reg_val;
      break;

    case B_DEC_SPEED:
      err = BestBasicBlockRead(handle, T_DEC_SPEED, &reg8_val, 1UL);
      *value = (b_int32) reg8_val;
      break;

    case B_DEC_LOCATION:
      err = BestBasicBlockRead(handle, T_DEC_MODE, &reg8_val, 1UL);

      if (reg8_val == B_MODE_IO)
        reg8_val = B_LOC_IO;
      else
        err = BestBasicBlockRead(handle, T_DEC_LOCATION, &reg8_val, 1UL);

      *value = (b_int32) reg8_val;
      break;

    case B_DEC_PREFETCH:
      err = BestBasicBlockRead(handle, T_DEC_PREFETCH, &reg8_val, 1UL);
      *value = (b_int32) reg8_val;
      break;

    default:
      B_FCT_PARAM_ERROR(2, "Invalid property");
    } /*lint !e788 ... not all enums used */
  }

  B_ERRETURN(err);
}


#ifndef BEST_FIRMWARE
/* should be in firmware but requires more modules, not needed yet */

b_errtype EXPORT BestTargetDecoderProg(b_handletype handle,
    b_decodertype decoder_num)
{
  B_DECLARE_FUNCNAME ("BestTargetDecoderProg [tdprog]");
  B_TRY_VARS_NO_PROG;

  B_TRY_BEGIN
  {
    /* license checking */
    B_TRY_LICENSE(B_CAPABILITY_HOSTINT);
    B_TRY(valid_decoder_num(handle, (b_int32)decoder_num));

    if (Best16BitRegisterFile(handle))
    {
      b_int8 decoder = (b_int8) decoder_num;
      /* TODO: store the location and return correct value of I/O mode in
       * property get */

      B_TRY(BestBasicCommand(handle, CMD_EXEFHIF_TDEC_PROG,
          &decoder, IN_EXEFHIF_TDEC_PROG, /* -> in */
          NULL, NULL));         /* <- out */
    }
    else
    {
      b_int8 zw8;
      b_int32 zw32;
      b_int32 size;
      b_errtype err;
      zw8 = (b_int8) decoder_num;

      if (zw8 == 0x07)
      {
        /* rom decoder */
        B_TRY(BestTargetDecoderPropGet(handle, B_DEC_SIZE, &size));

        if (size != 0)
        {
          B_TRY(BestBoardPropGet(handle, B_BOARD_ROMUSAGE, &zw32));

          if (zw32 == B_ROMUSAGE_INTERNAL)
	  { /* somebody forgot the braces here */
            B_TRY_ERROR(B_E_ROM_INTERNAL);
	  }
        }
      }

      err = BestBasicBlockWrite(handle, T_DEC_PROG, &zw8, 1UL);

      if (err == B_E_FUNC)
      {
        /* an error occured, so get the result code */
        B_TRY(BestBasicBlockRead(handle, T_DEC_RESULT, &zw8, 1UL));

        if (zw8 > 0x09)
        {
          B_TRY_ERROR(B_E_DEC_CHECK);
        }
        else
        {
          B_TRY_ERROR(decoder_result[zw8]);
        }
      }
      else
      {
        B_TRY_FAIL(err);
      }
    }
  }

  B_ERRETURN(B_TRY_RET);
}

#endif  /* BEST_FIRMWARE */


/* Program the preparation register into the power up properties */
b_errtype EXPORT BestTargetDecoderPowerUpProg(b_handletype handle,
    b_decodertype decoder_num)
{
  B_DECLARE_FUNCNAME ("BestTargetDecoderPowerUpProg [tdpuprog]");
  B_TRY_VARS_NO_PROG;

  B_TRY_BEGIN
  {
    /* license checking */
    B_TRY_LICENSE(B_CAPABILITY_HOSTINT);
    B_TRY(valid_decoder_num(handle, (b_int32)decoder_num));

    if (Best16BitRegisterFile(handle))
    {
      b_int8 decoder = (b_int8) decoder_num;
      /* TODO: store the location and return correct value of I/O mode in
       * property get */
      B_TRY(BestBasicCommand(handle, CMD_EXE_TDEC_PROGPU,
          &decoder, IN_EXE_TDEC_PROGPU, /* -> in */
          NULL, NULL));         /* <- out */
    }
    else
    {
      B_TRY_ERROR(B_E_NOT_E2925A);
    }
  }

  B_ERRETURN(B_TRY_RET);
}

b_errtype EXPORT BestTargetDecoderRead(b_handletype handle,
    b_decodertype decoder_num)
{
  B_DECLARE_FUNCNAME ("BestTargetDecoderRead [tdread]");
  b_errtype err;
  B_TRY_VARS_NO_PROG;

  B_TRY_BEGIN
  {
    /* license checking */
    B_LICENSECHECK(B_CAPABILITY_HOSTINT);
    B_TRY(valid_decoder_num(handle, (b_int32)decoder_num));

    if (Best16BitRegisterFile(handle))
    {
      b_int8 decoder = (b_int8) decoder_num;
      B_TRY(BestBasicCommand(handle, CMD_EXE_TDEC_READ,
          &decoder, IN_EXE_TDEC_READ, /* -> in */
          NULL, NULL));         /* <- out */
    }

    else
    {
      b_int8 zw8 = (b_int8) decoder_num;
      B_TRY(BestBasicBlockWrite(handle, T_DEC_READ, &zw8, 1UL));
    }
  }

  B_ERRETURN(B_TRY_RET);
}


/* Read the power up propertie into the preparation register */
b_errtype EXPORT BestTargetDecoderPowerUpRead(b_handletype handle,
    b_decodertype decoder_num)
{
  B_DECLARE_FUNCNAME ("BestTargetDecoderPowerUpRead [tdpuread]");
  b_errtype err;
  B_TRY_VARS_NO_PROG;
  b_int8 decoder = (b_int8) decoder_num;
  B_TRY_BEGIN
  {
    /* license checking */
    B_TRY_FAIL(BestIs2925(handle) ? B_E_NOT_E2925A : B_E_OK);
    B_LICENSECHECK(B_CAPABILITY_HOSTINT);
    B_TRY(valid_decoder_num(handle, (b_int32)decoder_num));


    B_TRY(BestBasicCommand(handle, CMD_EXE_TDEC_READPU,
        &decoder, IN_EXE_TDEC_READPU, /* -> in */
        NULL, NULL));           /* <- out */
  }

  B_ERRETURN(B_TRY_RET);
}


#ifndef BEST_FIRMWARE

/* convenience function, programs target decoder 1 or 2 */
b_errtype EXPORT BestTargetDecoder1xProg(
    b_handletype handle,
    b_decodertype decoder_num,
    b_int32 mode,
    b_int32 size,
    b_int32 base,
    b_int32 speed,
    b_int32 location,
    b_int32 prefetch
)
{
  B_TRY_VARS_NO_PROG;

  B_TRY_BEGIN
  {
    /* license checking */
    B_TRY_LICENSE(B_CAPABILITY_EXERCISER);
    B_TRY(valid_decoder_num(handle, (b_int32)decoder_num));
    B_TRY(BestTargetDecoderPropDefaultSet(handle, decoder_num));

    if (Best16BitRegisterFile(handle))
    {
      B_TRY(BestTargetDecoderPropSet(handle, B_DEC_BEHAVIOR, (b_int32)B_BEH_NORMAL));
      B_TRY(BestTargetDecoderPropSet(handle, B_DEC_BASEADDR_HI, 0L));
      B_TRY(BestTargetDecoderPropSet(handle, B_DEC_RESOURCE, (b_int32)B_RES_DATA));
      B_TRY(BestTargetDecoderPropSet(handle, B_DEC_RESBASE, 0L));
      B_TRY(BestTargetDecoderPropSet(handle, B_DEC_RESSIZE, 65536L));
    }

    /* program old decoder properties */
    B_TRY(BestTargetDecoderPropSet(handle, B_DEC_MODE, mode));
    B_TRY(BestTargetDecoderPropSet(handle, B_DEC_SIZE, size));
    B_TRY(BestTargetDecoderPropSet(handle, B_DEC_BASEADDR, base));
    B_TRY(BestTargetDecoderPropSet(handle, B_DEC_SPEED, speed));
    B_TRY(BestTargetDecoderPropSet(handle, B_DEC_LOCATION, location));
    B_TRY(BestTargetDecoderPropSet(handle, B_DEC_PREFETCH, prefetch));
    B_TRY(BestTargetDecoderProg(handle, decoder_num));
  }

  B_ERRETURN(B_TRY_RET);
}

#endif  /* BEST_FIRMWARE */


b_errtype EXPORT BestPCIConfigCheck(b_handletype handle)
{
  B_DECLARE_FUNCNAME ("BestPCIConfigCheck [cfgcheck]");
  b_int32 offset, value, mask, size;
  b_int32 minsize = 0;
  b_int32 maxsize = 0;
  b_int8 dacbase = 0;
  B_TRY_VARS_NO_PROG;

  B_TRY_BEGIN
  {
    for (offset = 0x00; offset < 0x40; offset += 4)
    {
      B_TRY(BestConfRegGet(handle, offset, &value));
      B_TRY(BestConfRegMaskGet(handle, offset, &mask));

      if (offset < 0x10)
      {
      }

      else if (offset < 0x28)
      {
        size = mask2size(mask);
        BestLastErrorParamSet(handle, B_ERRPAR_1, mask);
        BestLastErrorParamSet(handle, B_ERRPAR_2, offset);
        B_TRY_FAIL(size == B_DEC_INVALID ? B_E_CONFIG_MASK_INVALID : B_E_OK);
        BestLastErrorParamSet(handle, B_ERRPAR_1, value);

        if (dacbase)
        {
          minsize = 0;
          maxsize = 31;
        }
        else
        {
          /* base address registers */
          if (value & 0x00000001)
          {
            /* IO decoder */
            minsize = 2;
            maxsize = 16;
          }

          else
          {
            /* memory decoder */
            switch (value & 0x00000006)
            {
            case 0:
              minsize = 4;
              maxsize = 31;
              break;

            case 2:
              minsize = 4;
              maxsize = 20;
              break;

            case 4:
              minsize = 4;
              maxsize = 32;
              break;

            default:
              BestLastErrorParamSet(handle, B_ERRPAR_1, value);
              B_TRY_ERROR(B_E_CONFIG_VALUE_INVALID);
            }
          }
        }

	B_TRY_FAIL(
	(((1UL << ((int)size)) - 1) & value) ==
	(((1UL << ((int)minsize)) - 1) & value) ?
	  	B_E_OK : B_E_CONFIG_VALUE_INVALID);
        /* is this a good error message ? CZ */
        BestLastErrorParamSet(handle, B_ERRPAR_1, (b_int32)B_PARAM_DECODER_STD);
        BestLastErrorParamSet(handle, B_ERRPAR_2, (b_int32)B_DEC_SIZE);
        BestLastErrorParamSet(handle, B_ERRPAR_3, minsize);
        BestLastErrorParamSet(handle, B_ERRPAR_4, maxsize);
        BestLastErrorParamSet(handle, B_ERRPAR_5, size);
        B_TRY_FAIL((size < minsize) || (size > maxsize) ? B_E_RANGE : B_E_OK);
      }
    }
  }

  B_ERRETURN(B_TRY_RET);
}

#endif  /* E2921A_MIN */

